// Game_Music_Emu 0.4.0. http://www.slack.net/~ant/

#include "Spc_Emu.h"

#include <stdlib.h>
#include <string.h>
#include "blargg_endian.h"

/* Copyright (C) 2004-2006 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for
more details. You should have received a copy of the GNU Lesser General
Public License along with this module; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#include "blargg_source.h"

static Music_Emu* new_spc_emu() { return BLARGG_NEW Spc_Emu; }

gme_type_t_ const gme_spc_type [1] = { &new_spc_emu, &Spc_Emu::read_info };

Spc_Emu::Spc_Emu( double gain )
{
	apu.set_gain( gain );
	blargg_verify_byte_order();
}

Spc_Emu::~Spc_Emu() { }

const char** Spc_Emu::voice_names() const
{
	static const char* names [] = {
		"DSP 1", "DSP 2", "DSP 3", "DSP 4", "DSP 5", "DSP 6", "DSP 7", "DSP 8"
	};
	return names;
}

void Spc_Emu::mute_voices( int m )
{
	Music_Emu::mute_voices( m );
	apu.mute_voices( m );
}

blargg_err_t Spc_Emu::set_sample_rate_( long sample_rate )
{
	if ( sample_rate != native_sample_rate )
	{
		RETURN_ERR( resampler.buffer_size( native_sample_rate / 20 * 2 ) );
		resampler.time_ratio( (double) native_sample_rate / sample_rate, 0.9965 );
	}
	return 0;
}

blargg_err_t Spc_Emu::load( Data_Reader& in )
{
	header_t h;
	RETURN_ERR( in.read( &h, sizeof h ) );
	return load( h, in );
}

blargg_err_t Spc_Emu::load( const header_t& h, Data_Reader& in )
{
	if ( in.remain() < Snes_Spc::spc_file_size - (int) sizeof h )
		return "Not an SPC file";
	
	if ( strncmp( h.tag, "SNES-SPC700 Sound File Data", 27 ) != 0 )
		return "Not an SPC file";
	
	long remain = in.remain();
	spc_data_size = remain + sizeof h;
	RETURN_ERR( spc_data.resize( max( spc_data_size, (long) trailer_offset ) ) );
	
	set_track_count( 1 );
	set_voice_count( Snes_Spc::voice_count );
	
	memcpy( spc_data.begin(), &h, sizeof h );
	return in.read( &spc_data [sizeof h], remain );
}

void Spc_Emu::start_track_( int track )
{
	Music_Emu::start_track_( track );
	
	resampler.clear();
	if ( !apu.load_spc( spc_data.begin(), spc_data_size ) )
		apu.clear_echo();
	else
		check( false );
}

void Spc_Emu::skip_( long count )
{
	count = long (count * resampler.ratio()) & ~1;
	
	count -= resampler.skip_input( count );
	if ( count > 0 )
		if ( apu.skip( count ) ) { } // ignore error
	
	// eliminate pop due to resampler
	const int resampler_latency = 64;
	sample_t buf [resampler_latency];
	play_( resampler_latency, buf );
}

void Spc_Emu::play_( long count, sample_t* out )
{
	require( track_count() ); // file must be loaded
	
	if ( sample_rate() == native_sample_rate )
	{
		if ( apu.play( count, out ) )
			log_error();
		return;
	}
	
	long remain = count;
	while ( remain > 0 )
	{
		remain -= resampler.read( &out [count - remain], remain );
		if ( remain > 0 )
		{
			long n = resampler.max_write();
			if ( apu.play( n, resampler.buffer() ) )
				log_error();
			resampler.write( n );
		}
	}
	
	assert( remain == 0 );
}

// Track info

static void get_spc_xid6( byte const* begin, long size, track_info_t* out )
{
	// header
	byte const* end = begin + size;
	if ( size < 8 || memcmp( begin, "xid6", 4 ) )
	{
		check( false );
		return;
	}
	long info_size = get_le32( begin + 4 );
	byte const* in = begin + 8; 
	if ( end - in > info_size )
	{
		dprintf( "Extra data after SPC xid6 info\n" );
		end = in + info_size;
	}
	
	int year = 0;
	char copyright [256 + 5];
	char* copyright_end = &copyright [5];
	int copyright_len = 0;
	int const year_len = 5;
	
	while ( end - in >= 4 )
	{
		// header
		int id   = in [0];
		int data = in [3] * 0x100 + in [2];
		int type = in [1];
		int len  = type ? data : 0;
		in += 4;
		if ( len > end - in )
		{
			check( false );
			break; // block goes past end of data
		}
		
		// handle specific block types
		char* field = 0;
		switch ( id )
		{
			case 0x01: field = out->song;      break;
			case 0x02: field = out->game;      break;
			case 0x03: field = out->author;    break;
			case 0x04: field = out->dumper;    break;
			case 0x07: field = out->comment;   break;
			case 0x14: year = data;            break;
			
			//case 0x30: // intro length
			// Many SPCs have intro length set wrong for looped tracks, making it useless
			/*
			case 0x30:
				check( len == 4 );
				if ( len >= 4 )
				{
					out->intro_length = get_le32( in ) / 64;
					if ( out->length > 0 )
					{
						long loop = out->length - out->intro_length;
						if ( loop >= 2000 )
							out->loop_length = loop;
					}
				}
				break;
			*/
			
			case 0x13:
				copyright_len = min( len, (int) sizeof copyright - year_len );
				memcpy( &copyright [year_len], in, copyright_len );
				break;
			
			default:
				if ( id < 0x01 || (id > 0x07 && id < 0x10) ||
						(id > 0x14 && id < 0x30) || id > 0x36 )
					dprintf( "Unknown SPC xid6 block: %X\n", (int) id );
				break;
		}
		if ( field )
		{
			check( type == 1 );
			out->copy_field( field, (char*) in, len );
		}
		
		// skip to next block
		in += len;
		
		// blocks are supposed to be 4-byte aligned with zero-padding...
		byte const* unaligned = in;
		while ( (in - begin) & 3 && in < end )
		{
			if ( *in++ != 0 )
			{
				// ...but some files have no padding
				in = unaligned;
				dprintf( "SPC info tag wasn't properly padded to align\n" );
				break;
			}
		}
	}
	
	char* p = &copyright [year_len];
	if ( year )
	{
		*--p = ' ';
		for ( int n = 4; n--; )
		{
			*--p = year % 10 + '0';
			year /= 10;
		}
		copyright_len += year_len;
	}
	if ( copyright_len )
		out->copy_field( out->copyright, p, copyright_len );
	
	check( in == end );
}

static void get_spc_info( Spc_Emu::header_t const& h, byte const* xid6, long xid6_size,
		track_info_t* out )
{
	// decode length (can be in text or binary format, sometimes ambiguous ugh)
	long len_secs = 0;
	for ( int i = 0; i < 3; i++ )
	{
		unsigned n = h.len_secs [i] - '0';
		if ( n > 9 )
		{
			// ignore single-digit text lengths
			// (except if author field is present and begins at offset 1, ugh)
			if ( i == 1 && (h.author [0] || !h.author [1]) )
				len_secs = 0;
			break;
		}
		len_secs *= 10;
		len_secs += n;
	}
	if ( !len_secs || len_secs > 0x1FFF )
		len_secs = get_le16( h.len_secs );
	if ( len_secs < 0x1FFF )
		out->length = len_secs * 1000;
	
	if ( xid6_size )
		get_spc_xid6( xid6, xid6_size, out );
	
	int offset = (h.author [0] < ' ' || unsigned (h.author [0] - '0') <= 9);
	out->copy_field( out->author, &h.author [offset], sizeof h.author - offset );
	
	GME_COPY_FIELD( h, out, song );
	GME_COPY_FIELD( h, out, game );
	GME_COPY_FIELD( h, out, dumper );
	GME_COPY_FIELD( h, out, comment );
	out->copy_field( out->system, "Super Nintendo" );
}

blargg_err_t Spc_Emu::track_info( track_info_t* out, int track ) const
{
	if ( Music_Emu::track_info( out, track ) ) { }
	get_spc_info( header(), trailer(), trailer_size(), out );
	return out->error();
}

blargg_err_t Spc_Emu::read_info( Data_Reader& in, track_info_t* out, int track )
{
	header_t h;
	RETURN_ERR( in.read( &h, sizeof h ) );
	
	long const xid6_skip = 0x10200 - sizeof h;
	long xid6_size = in.remain() - xid6_skip;
	blargg_vector<byte> xid6;
	if ( xid6_size > 0 )
	{
		RETURN_ERR( xid6.resize( xid6_size ) );
		RETURN_ERR( in.skip( xid6_skip ) );
		RETURN_ERR( in.read( xid6.begin(), xid6.size() ) );
	}
	
	out->init();
	get_spc_info( h, xid6.begin(), xid6_size, out );
	return out->error();
}
